#define WITH_TEXCOORD_OPS
/*=============================================================================
	FixedEmuTypes.h: Duke Forever Direct3D9 support
	Copyright 1997-2003 3D Realms, Inc. All Rights Reserved.
	
	Revision history:
		Created by John Pollard
=============================================================================*/
#ifndef FIXED_EMU_TYPES_H
#define FIXED_EMU_TYPES_H

#define GAUSSIAN_OLD
#define OLD_TRANSPARENT_BUMPS

#define FEMU_MAX_OPS						(48)
#define FEMU_MAX_ARGS						(256)
#define	FEMU_MAX_SAMPLERS					(16)
#define	FEMU_MAX_TEXCOORD_OPS				(16)
#define FEMU_MAX_RESULTS					(FEMU_MAX_OPS)			// Temp registers to hold results of each operation
#define MAX_FIXEDEMU_LIGHTS					(16)
#define FEMU_MAX_CONSTS						(16)
#define FEMU_MAX_MATRICES					(16)

#define NULL_HANDLE							(255)
//#define INVALID_HANDLE						(254)
											
//
//	FEMU_OP		(Operation)
//

#define FEMU_OP_Disable						(0)
#define FEMU_OP_Nop							(1)
#define	FEMU_OP_TexCoords					(2)			// Load TexCoords from In.TexCoords[Arg[0]]
#define FEMU_OP_Diffuse						(3)			// Loads diffuse color 
#define FEMU_OP_Texture						(4)			// Load Texture data
#define FEMU_OP_Add							(5)			// Add Arg[0]+Arg[1]
#define FEMU_OP_AddSmooth					(6)
#define FEMU_OP_Subtract					(7)
#define FEMU_OP_Dot3Biased					(8)
#define FEMU_OP_Dot3						(9)
#define FEMU_OP_AddSigned2X					(10)
#define FEMU_OP_Min							(11)
#define FEMU_OP_Max							(12)
#define FEMU_OP_Exp							(13)
#define FEMU_OP_Cross						(14)
#define FEMU_OP_AddPreMultDiffAlpha			(15)
#define FEMU_OP_Mul							(16)			// Mul Arg[0]*Arg[1]
#define FEMU_OP_Mul2X						(17)			// Mul Arg[0]*Arg[1]*2
#define FEMU_OP_Mul4X						(18)			// Mul Arg[0]*Arg[1]*4
#define FEMU_OP_Mul_Alpha					(19)			// Lerp( Arg[0], Arg[0]*Arg[1], Arg[0].a )
#define	FEMU_OP_BlendDiffuseAlpha			(20)
#define FEMU_OP_DeferredBasePass			(21)			//
#define FEMU_OP_DeferredLight				(22)
#define FEMU_OP_Parallax					(23)		// Shift UV's to simulate parallaxing effect
#define	FEMU_OP_AlphaTest					(24)		// Will early out of a pixel shader if the alphatest fails
#define FEMU_OP_Lerp						(25)
#define FEMU_OP_SetAlpha					(26)		// Override alpha channel with Arg 0
#define	FEMU_OP_Const						(27)
#define	FEMU_OP_ReturnValue					(28)
#define FEMU_OP_FakeBumpMap					(29)
#define FEMU_OP_BumpCubeEnvMap				(30)
#define FEMU_OP_BumpWaterReflection			(31)
#define FEMU_OP_MultiPassLight				(32)
#define FEMU_OP_Bloom						(33)
#define FEMU_OP_ForwardBasePass				(34)		// Base pass for old style forward renderer
#define FEMU_OP_DeferredBasePass_ThermalBS	(35)
#define FEMU_OP_Unused						(36)		// Was shadowvolume
#define FEMU_OP_FogVolumePass0				(37)
#define FEMU_OP_FogVolumePass1				(38)
#define FEMU_OP_HeightFog_Forward			(39)
#define FEMU_OP_HeightFog_Deferred			(40)
#define FEMU_OP_DepthOfField0				(41)
#define FEMU_OP_PixelMotionBlur0			(42)		// Motion blur first pass, initializes velocity buffer
#define FEMU_OP_Unused2						(44)
#define FEMU_OP_DeferredAmbient_Bumps		(45)
#define FEMU_OP_DeferredFog					(46)
#define FEMU_OP_DeferredFSAA				(47)
#define FEMU_OP_DeferredPosOnly				(48)
#define FEMU_OP_ThermalPostPass1			(49)
#define FEMU_OP_ThermalPostPass2			(50)
#define FEMU_OP_DeferredBasePass_Thermal	(51)		// Non bumpshader version of base pass thermal
#define FEMU_OP_TexScaleShift				(52)		// Scales and shifts existing texture coords (in pixel shader, so doesn't rely on number of tex registers)
#define	FEMU_OP_DeferredBasePass_Detail		(53)
#define FEMU_OP_DistanceColor				(54)		// Produces a color based on distance form camera
#define FEMU_OP_Mul2X_Alpha					(55)
#define FEMU_OP_ShadowMapPass				(56)
#define FEMU_OP_1DConvolutionFilter			(57)		// 1d seperable convolution filter
#define FEMU_OP_CalcLum						(58)
#define FEMU_OP_CalcAdaptedLum				(59)
#define FEMU_OP_FinalHDRPass				(60)
#define FEMU_OP_CalcAvg						(61)
#define FEMU_OP_StarBlur					(62)
#define FEMU_OP_StarMerge					(63)
#define FEMU_OP_RestoreZBuffer				(64)
#define FEMU_OP_LightShaft					(65)
#define FEMU_OP_MergeShadows				(66)
#define FEMU_OP_SoftEdge					(67)
#define FEMU_OP_ForwardDistanceFog			(68)		// Distance fog, for forward rendered objects
#define FEMU_OP_TextureProj5				(69)
#define FEMU_OP_GenShadowMapTestResults		(70)
#define FEMU_OP_AmbientOcclusion			(71)
#define FEMU_OP_BlurAmbientOcclusion		(77)
#define FEMU_OP_Foliage						(78)
#define FEMU_OP_HeightFog_ForwardNew		(79)
#define FEMU_OP_AlphaNormalize				(80)
#define FEMU_OP_FinalBloomPass				(81)
#define FEMU_OP_PostProcessor				(82)
#define FEMU_OP_DepthOfField1				(83)
#define FEMU_OP_FinalizeAmbientOcclusion	(84)
#define FEMU_OP_Mask						(85)
#define FEMU_OP_AvgDepthBuffer0			    (86)
#define FEMU_OP_AvgDepthBuffer1			    (87)
#define FEMU_OP_GenerateFocalPlane			(88)
#define FEMU_OP_OverlayBumpDiffuse			(89)
#define FEMU_OP_FakeDirectionalLight		(90)
#define FEMU_OP_EmissiveBloom				(91)
#define FEMU_OP_BinkShader					(92)
#define FEMU_OP_ScreenFillFromTexture		(93)
#define FEMU_OP_ComputeNormalSpec			(94)
#define FEMU_OP_RimLightEffect				(95)
#define FEMU_OP_FinalHDRPass_Emissive		(96)
#define FEMU_OP_FinalHDRPass_Diffuse		(97)
#define FEMU_OP_BloomXBox					(98)
#define FEMU_OP_DeferredBasePass_HiJackOp  (99)
#define FEMU_OP_BumpWaterReflection_NoSNoA	(100)
#define FEMU_OP_DeferredLight_HiJackOp		(101)
#define FEMU_OP_BloomPrePass				(102) // This is only here to make the OfflineShaderCompiler build
#define FEMU_OP_UberPostGather				(103)
#define FEMU_OP_UberPostBlend				(104)
#define FEMU_OP_InteractEmissive			(105)
#define FEMU_OP_DeferredBasePassInteractTint (106)
#define FEMU_OP_DeferredBasePassHoloDuke	(107)
#define FEMU_OP_FXAA				        (108)

//
//	FEMU_TCI	(Texture Coordinate Index)
//

#define FEMU_TCI_TexCoord0					(0)
#define FEMU_TCI_TexCoord1					(1)
#define FEMU_TCI_TexCoord2					(2)
#define FEMU_TCI_TexCoord3					(3)
#define FEMU_TCI_TexCoord4					(4)
#define FEMU_TCI_TexCoord5					(5)
#define FEMU_TCI_TexCoord6					(6)
#define FEMU_TCI_TexCoord7					(7)
#define FEMU_TCI_CameraPos					(8)
#define FEMU_TCI_WorldPos					(9)
#define FEMU_TCI_CameraReflection			(10)
#define FEMU_TCI_CameraNormal				(11)
#define FEMU_TCI_Nop						(12)

//
//	FEMU_TT		(Texture Type)
//

#define FEMU_TT_1D							(0)
#define FEMU_TT_2D							(1)
#define FEMU_TT_3D							(2)
#define FEMU_TT_CUBE						(3)

//
//	FEMU_TF		(Transform Flags)
//

#define FEMU_TF_None						(0)
#define FEMU_TF_Count2						(1)
#define FEMU_TF_Count3						(2)
#define FEMU_TF_Count4						(3)
#define FEMU_TF_Count2_Persp				(4)
#define FEMU_TF_Count3_Persp				(5)
#define FEMU_TF_Count4_Persp				(6)

#define SHADOW_FILTER_NORMAL				(0)
#define SHADOW_FILTER_ATI					(1)
#define SHADOW_FILTER_NVIDIA				(2)
#define SHADOW_FILTER_XBOX					(3)



#endif // FIXED_EMU_TYPES_H

struct VS_INPUT
{
    float4 vPosition		 : POSITION;
    float4 vColor		 : COLOR;
    float4 vNormal		 : NORMAL;
    float4 vTexCoords[8]  : TEXCOORD0;
};
struct VS_OUTPUT
{
    float4 vPosition      : POSITION;
    float4 vColor		 : COLOR;
    float4 vTexCoords[1] : TEXCOORD0;
};
matrix	 LocalToWorld	: register (vs, c0);
matrix	 LocalToCamera	: register (vs, c4);
matrix	 LocalToScreen	: register (vs, c8);
#ifdef WITH_NV_STEREO
matrix	 ScreenToCamera	: register (vs, c12);
#endif
matrix    PrevLocalToScreen;
float3x4	 ProjectorTexToLocal;
vector    AmbientLighting;
vector	 CameraPos;
float4    g_Time;
float4    g_MiscFoliage;
matrix   StageMatrices[1];
float4     Consts[FEMU_MAX_CONSTS];
float4x3   UserMatrices[FEMU_MAX_MATRICES];
matrix     CameraToShadowCubeMap;
#define bHasVertexColor        (1)
#define bHasVertexNormal       (1)
#define bMeshLighting          (1)
#define bTwoSidedLighting      (1)
#define bVertexAmbientLighting (0)
#define bVertexLighting        (1)
#define bProjectorLighting     (0)
#define bBillboardPolys        (0)
#define bModulateAmbient       (0)
#define bHasTexCoordOps       (1)
#define NumLights              (1)
struct TexCoordOp
{
    int      StreamIndex;
    int      TransformFlag;
};
static const TexCoordOp TexCoordOps[1]={0,0};
struct LightInfo
{
    // For old dx7 style hw lights
    vector       Location;
    vector       Color;

    // Per pixel lightinfo
    matrix       LocalToLightAtten3D;
    matrix       LocalToFilter;
};
LightInfo	Lights[1];
//================================================================================
//	ShaderRuntime_VS
//	Copyright 2007 3D Realms, Inc. All Rights Reserved.
//	John Pollard
//================================================================================

#define ARG(Index)			(Args[ArgStart+Index])
#define VARYARG(Index)		(Args[VaryArgStart+Index])
#define PACK_CAMERA_Z(Z)	((Z)*0.2f)
#define UNPACK_CAMERA_Z(Z)	((Z)*(1.0f/0.2f))
//#define PACK_CAMERA_Z(Z)	(Z)

// FIXME: Generate this on the fly?
vector	OneOverViewportSize;

#ifdef WITH_NV_STEREO
	sampler NvStereoFixTex : register(vs, s0);
#endif

float4 NvMonoToStereoClipPosVS(float4 InMonoClipPos)
{
#ifdef WITH_NV_STEREO
	if (InMonoClipPos.w == 1.0) {
		return InMonoClipPos;
	}

	// stereoParams.x is separation
	// stereoParams.y is convergence
	float2 stereoParams = tex2Dlod(NvStereoFixTex, float4(1.0f / 16, 0.5, 0, 0)).xy;
	float4 OutStereoClipPos = InMonoClipPos;
	
	// Perform the stereo transform for shaders that skipped it. 
	// The stereo transform is StereoClipPos.x += (Separation * (StereoClipPos.w - Convergence));
	OutStereoClipPos.x += (stereoParams.x * (InMonoClipPos.w - stereoParams.y));

	return OutStereoClipPos;
#else
	return InMonoClipPos;
#endif
}

//================================================================================
//	HandleVertexLighting
//	Handle dx8 style hw lights
//================================================================================
float3 HandleVertexLighting(float3 Diffuse, float4 LocalPos, float3 Normal)
{
	float4 WorldPos		= mul(LocalPos, LocalToWorld);				// Rotate vertex position into world
	float3 WorldNormal	= mul(Normal, (float3x3)LocalToWorld);		// Rotate vertex normal into world
	float3 TotalLight	= float3(0,0,0);
	
	for (int i=0; i< NumLights; i++)
	{
		float3	VertToLight	= normalize((Lights[i].Location - WorldPos).rgb);		// Get vertex to light vector
		float4	LightPos	= mul(LocalPos, Lights[i].LocalToLightAtten3D);
		//float	Dot3		= bTwoSidedLighting ? abs(dot(VertToLight, WorldNormal)) : max(dot(VertToLight, WorldNormal), 0);
		float	Dot3		= bTwoSidedLighting ? 1.0f : max(dot(VertToLight, WorldNormal), 0);
		float	Atten		= 1.0f - clamp(length(LightPos.rgb*2-1), 0, 1);

		TotalLight += (Lights[i].Color * Dot3 * Atten).rgb;		// Attenuate
	}
	
	if (bVertexAmbientLighting && bHasVertexColor)
		TotalLight += Diffuse;							// Add vertex color in
	else
		TotalLight *= Diffuse;
	
	if (bModulateAmbient)
		TotalLight *= AmbientLighting;
	else
		TotalLight += AmbientLighting;					// Add in ambient
		
	return TotalLight;
}

static float3 FoliageLocal;

//================================================================================
//	HandleBillboardPolys
//================================================================================
float4 HandleBillboardPolys(const VS_INPUT In, float4 Position)
{
#if (bHasTexCoordOps > 0)
	float Size = 1-saturate((Position.z-g_MiscFoliage.x) / g_MiscFoliage.y);
	
	Size *= g_Time.y;
	
	Position = In.vPosition;

	float3 CamPos = float3(g_Time.z, g_Time.w, Position.z);
	
	float3 Vector = (Position.xyz - CamPos);
	float Length = length(Vector);
	
	Vector /= Length;
	
	Length = 1-saturate(Length/150);
	
	matrix Trans = transpose(LocalToCamera);
	
	Position.x += sin((g_Time.x + Position.x*10.2f + Position.y*15.2f) * 0.06f) * 4.0f * In.vTexCoords[7].z;
	Position.y += cos((g_Time.x + Position.x*10.2f + Position.y*15.2f) * 0.06f) * 4.0f * In.vTexCoords[7].z;

	float3 XVector = Trans[0];
	float3 ZVector = -Trans[1];
	
	ZVector.z -= 2.0f;
	ZVector = normalize(ZVector);
	
	Position.xyz += XVector * In.vTexCoords[7].x * Size;
	Position.xyz += ZVector * In.vTexCoords[7].y * Size;
	
	Position.xyz += Vector * Length * 20.0f * In.vTexCoords[7].z;

	FoliageLocal = Position.xyz;
	
	Position = mul(float4(Position.xyz, 1), LocalToScreen);
#endif	
	return Position;
}

#ifdef WITH_TEXCOORD_OPS

void HandleTransformFlag(const TexCoordOp TexOp, const int Index, inout VS_OUTPUT Out)
{
#if (bHasTexCoordOps > 0)

	if (TexOp.TransformFlag != FEMU_TF_None)
		Out.vTexCoords[Index] = mul(Out.vTexCoords[Index], StageMatrices[Index]);
		
#endif
}

//================================================================================
//	HandleTexCoordOp
//================================================================================
void HandleTexCoordOp_StreamIndex7(const TexCoordOp TexOp, const int Index, float3 LocalNormal, const VS_INPUT In, inout VS_OUTPUT Out)
{

#if (bHasTexCoordOps > 0)

	Out.vTexCoords[Index] = In.vTexCoords[TexOp.StreamIndex];
	
	// Make sure unused components are properly initialized
	if (TexOp.TransformFlag == FEMU_TF_Count2 || TexOp.TransformFlag == FEMU_TF_Count2_Persp)
	{
		Out.vTexCoords[Index].z = 1.0f;
		Out.vTexCoords[Index].w = 1.0f;
	}
	else if (TexOp.TransformFlag == FEMU_TF_Count3 || TexOp.TransformFlag == FEMU_TF_Count3_Persp)
		Out.vTexCoords[Index].w = 1.0f;
	else
	{
		Out.vTexCoords[Index].z = 1.0f;
		Out.vTexCoords[Index].w = 1.0f;
	}
	
	HandleTransformFlag(TexOp, Index, Out);
	
#endif	
}

void HandleTexCoordOp_CameraPos(const TexCoordOp TexOp, const int Index, float3 LocalNormal, const VS_INPUT In, inout VS_OUTPUT Out)
{
#if (bHasTexCoordOps > 0)
#ifdef WITH_NV_STEREO
	// Use the computed vPosition field, then convert to correct stereo, then convert back to eye space.
	Out.vTexCoords[Index] = Out.vPosition;
	Out.vTexCoords[Index].x = NvMonoToStereoClipPosVS(Out.vTexCoords[Index]).x;
	Out.vTexCoords[Index] = mul(Out.vTexCoords[Index], ScreenToCamera);
	Out.vTexCoords[Index] /= Out.vTexCoords[Index].w;
#else
	Out.vTexCoords[Index] = mul(float4(In.vPosition.xyz,1), LocalToCamera);
#endif

 	HandleTransformFlag(TexOp, Index, Out);	
#endif	
}

void HandleTexCoordOp_WorldPos(const TexCoordOp TexOp, const int Index, float3 LocalNormal, const VS_INPUT In, inout VS_OUTPUT Out)
{
#if (bHasTexCoordOps > 0)

	Out.vTexCoords[Index] = mul(float4(In.vPosition.xyz,1), LocalToWorld);
	
	HandleTransformFlag(TexOp, Index, Out);
	
#endif	
}

void HandleTexCoordOp_CameraReflection(const TexCoordOp TexOp, const int Index, float3 LocalNormal, const VS_INPUT In, inout VS_OUTPUT Out)
{
#if (bHasTexCoordOps > 0)

	// Put pos in camera space
	float4 TexCoords = mul(In.vPosition, LocalToCamera);	
	// Put normal in camera space
	LocalNormal = mul(LocalNormal, (float3x3)LocalToCamera);
	// Reflect vector by vertex normal and normalize
	TexCoords.xyz = reflect(normalize(TexCoords.xyz), normalize(LocalNormal));
	
	// Since we are now dealing with a normal, only rotate, don't translate
	if (TexOp.TransformFlag != FEMU_TF_None)
		TexCoords.xyz = mul(TexCoords, (float3x3)StageMatrices[Index]).xyz;

	Out.vTexCoords[Index] = TexCoords;

#endif	
}

void HandleTexCoordOp_CameraNormal(const TexCoordOp TexOp, const int Index, float3 LocalNormal, const VS_INPUT In, inout VS_OUTPUT Out)
{
#if (bHasTexCoordOps > 0)

	float4 WorldCoords = mul(float4(In.vPosition.xyz,1), LocalToWorld);	
	
	//float3 WorldNormal = mul(In.vNormal, (float3x3)LocalToWorld);
	float3 WorldNormal = mul(LocalNormal, (float3x3)LocalToWorld);
	
	WorldNormal = normalize(WorldNormal);
	
	Out.vTexCoords[Index+0]		= WorldCoords;
	Out.vTexCoords[Index+1].xyz = WorldNormal;

#endif	
}

//================================================================================
//	HandleFSAACoords
//================================================================================
void HandleFSAACoords(const VS_INPUT In, int Index, int ConstIndex, inout VS_OUTPUT Out)
{
#if (bHasTexCoordOps > 0)
	float4 Tex = In.vTexCoords[0];
			
	float XOfs = Consts[ConstIndex].z;
	float YOfs = Consts[ConstIndex].w;
		
	Out.vTexCoords[Index+0] = float4(Tex.x, Tex.y, 1.0f, 1.0f);				// Center
	Out.vTexCoords[Index+1] = float4(Tex.x-XOfs, Tex.y-YOfs, 1.0f, 1.0f);	// Left Top
	Out.vTexCoords[Index+2] = float4(Tex.x+XOfs, Tex.y+YOfs, 1.0f, 1.0f);	// Right bottom
	Out.vTexCoords[Index+3] = float4(Tex.x+XOfs, Tex.y-YOfs, 1.0f, 1.0f);	// Right Top
	Out.vTexCoords[Index+4] = float4(Tex.x-XOfs, Tex.y+YOfs, 1.0f, 1.0f);	// Left Bottom
	Out.vTexCoords[Index+5] = float4(Tex.x-XOfs, Tex.y, Tex.x+XOfs, Tex.y);	// Left / Right
	Out.vTexCoords[Index+6] = float4(Tex.x, Tex.y-YOfs, Tex.x, Tex.y+YOfs);	// Top / Bottom
#endif	
}

//================================================================================
//	CalcTexToLocal
//================================================================================
float3x3 CalcTexToLocal(const VS_INPUT In)
{
#if (bProjectorLighting)

	return (float3x3)ProjectorTexToLocal;	
	
#else

	#if ((bHasVertexNormal == 1) && (bHasTexCoordOps > 0))	
		float3 XAxis = In.vTexCoords[3].xyz;
		float3 YAxis = bMeshLighting ? cross(In.vNormal, In.vTexCoords[3].xyz) * In.vTexCoords[0].z : In.vTexCoords[4].xyz;
		float3 ZAxis = In.vNormal.xyz;		
	#else 
		float3 XAxis = float3(0,-1,0);
		float3 YAxis = float3(0,0,1);
		float3 ZAxis = float3(1,0,0);
	#endif	
	
	return float3x3(XAxis, YAxis, ZAxis);
	
#endif
}



//================================================================================
//	HandleTexCoordOp
//================================================================================
void HandleTexCoordOp(const TexCoordOp TexOp, const int Index, float3 LocalNormal, const VS_INPUT In, inout VS_OUTPUT Out)
{
#if (bHasTexCoordOps > 0)
	//if (TexOp.StreamIndex == FEMU_TCI_Nop)		// This was a reserved tex coord
	//	return;
	
	if (TexOp.StreamIndex <= 7)
	{
		Out.vTexCoords[Index] = In.vTexCoords[TexOp.StreamIndex];
		
		// Make sure unused components are properly initialized
		if (TexOp.TransformFlag == FEMU_TF_Count2 || TexOp.TransformFlag == FEMU_TF_Count2_Persp)
		{
			Out.vTexCoords[Index].z = 1.0f;
			Out.vTexCoords[Index].w = 1.0f;
		}
		else if (TexOp.TransformFlag == FEMU_TF_Count3 || TexOp.TransformFlag == FEMU_TF_Count3_Persp)
			Out.vTexCoords[Index].w = 1.0f;
		else
		{
			Out.vTexCoords[Index].z = 1.0f;
			Out.vTexCoords[Index].w = 1.0f;
		}
	}
	else if (TexOp.StreamIndex == FEMU_TCI_CameraPos)				// From camera space
		Out.vTexCoords[Index] = mul(float4(In.vPosition.xyz,1), LocalToCamera);	
	else if (TexOp.StreamIndex == FEMU_TCI_WorldPos)				// From world space
		Out.vTexCoords[Index] = mul(float4(In.vPosition.xyz,1), LocalToWorld);	
	else if (TexOp.StreamIndex == FEMU_TCI_CameraReflection)		// From camera space reflected by normal
	{
		// Put pos in camera space
		float4 TexCoords = mul(In.vPosition, LocalToCamera);	
		// Put normal in camera space
		LocalNormal = mul(LocalNormal, (float3x3)LocalToCamera);
		// Reflect vector by vertex normal and normalize
		TexCoords.xyz = reflect(normalize(TexCoords.xyz), normalize(LocalNormal));
		
		// Since we are now dealing with a normal, only rotate, don't translate
		if (TexOp.TransformFlag != FEMU_TF_None)
			TexCoords.xyz = mul(TexCoords, (float3x3)StageMatrices[Index]).xyz;
	
		Out.vTexCoords[Index] = TexCoords;
		
		return;		// We already transformed the texcoords, so bail out
	}
	else if (TexOp.StreamIndex == FEMU_TCI_CameraNormal)
	{
	#if 1
		/*
		// Put pos in camera space
		float4 TexCoords = mul(In.vPosition, LocalToCamera);	
		// Put normal in camera space
		LocalNormal = mul(LocalNormal, (float3x3)LocalToCamera);
		// Reflect vector by vertex normal and normalize
		TexCoords.xyz = reflect(normalize(TexCoords.xyz), normalize(LocalNormal));
		
		// Since we are now dealing with a normal, only rotate, don't translate
		if (TexOp.TransformFlag != FEMU_TF_None)
			TexCoords.xyz = mul(TexCoords, (float3x3)StageMatrices[Index]).xyz;
	
		Out.vTexCoords[Index] = TexCoords;
		*/
		
		float4 WorldCoords = mul(float4(In.vPosition.xyz,1), LocalToWorld);	
		
		//float3 WorldNormal = mul(In.vNormal, (float3x3)LocalToWorld);
		float3 WorldNormal = mul(LocalNormal, (float3x3)LocalToWorld);
		
		WorldNormal = normalize(WorldNormal);
		
		Out.vTexCoords[Index+0]		= WorldCoords;
		Out.vTexCoords[Index+1].xyz = WorldNormal;

		return;		// We already transformed the texcoords, so bail out
	#else
		// The idea here would be to send the world normal down the vertex pipe.
		// Then in the pixel shader, sample the texel color from 5 axes, and then blend between
		//	them using the world space vertex normal (passed from here)
		
		float4 WorldCoords = mul(float4(In.vPosition.xyz,1), LocalToWorld);	
		
		float4 Axis1 = float4(WorldCoords.xz, 1, 1) * 0.001f;
		float4 Axis2 = float4(WorldCoords.yz, 1, 1) * 0.001f;
		float4 Axis3 = float4(WorldCoords.yx, 1, 1) * 0.001f;

		// float3 WorldNormal = mul(In.vNormal, (float3x3)LocalToWorld);
		float3 WorldNormal = mul(LocalNormal, (float3x3)LocalToWorld);
		
		WorldNormal = normalize(WorldNormal);
		
		float3 AxisAlpha;
		
		AxisAlpha.x = abs(dot(WorldNormal, float3(0,1,0)));
		AxisAlpha.y = abs(dot(WorldNormal, float3(1,0,0)));
		AxisAlpha.z = abs(dot(WorldNormal, float3(0,0,1)));
		
		//Out.vTexCoords[Index] = Axis1;
		
		//AxisAlpha = 1;
		
		//float2 Alpha1 = normalize(AxisAlpha.xy);
		//float2 Alpha2 = normalize(AxisAlpha.yz);
		
		Out.vTexCoords[Index] = float4(WorldNormal, 1);
		
		//Out.vTexCoords[Index] = Axis1*AxisAlpha.x + Axis2*AxisAlpha.y + Axis3*AxisAlpha.z;
		//Out.vTexCoords[Index] = Axis1*Alpha1.x + Axis2*Alpha1.y;
		//Out.vTexCoords[Index] = Out.vTexCoords[Index]*Alpha2.x + Axis3*Alpha2.y;
		
		
		/*
		if (Alpha1 > 0.1f || Alpha1 < -0.1)
			Out.vTexCoords[Index] = Axis1;
		else if (Alpha2 > 0.1f || Alpha2 < -0.1)
			Out.vTexCoords[Index] = Axis2;
		else 
			Out.vTexCoords[Index] = 0;
		*/
			
		return;
	#endif
	}
	
	if (TexOp.TransformFlag != FEMU_TF_None)
		Out.vTexCoords[Index] = mul(Out.vTexCoords[Index], StageMatrices[Index]);
#endif	
}

#endif

void DeferredBasePassVS(in VS_INPUT In, inout VS_OUTPUT Out, in matrix LocalToCamera, int Index, int arg_7, int arg_8)
{
	// const int Index = ARG(3);
		
	// Get camera space position
	float3 CameraSpacePos = mul(float4(In.vPosition.xyz, 1), LocalToCamera).xyz;

#if (/*(bHasVertexNormal == 1) &&*/ (bHasTexCoordOps > 0))
	float3x3 TexToLocal = CalcTexToLocal(In);

	float3x3 TexToCamera = mul(TexToLocal, (float3x3)LocalToCamera);

	// Output TexToCamera matrix
	Out.vTexCoords[Index+0].xyz = TexToCamera[0];
	Out.vTexCoords[Index+1].xyz = TexToCamera[1];
	Out.vTexCoords[Index+2].xyz = TexToCamera[2];
		
	// Stuff camera pos into w values
	Out.vTexCoords[Index+0].w = CameraSpacePos.x;
	Out.vTexCoords[Index+1].w = CameraSpacePos.y;
	Out.vTexCoords[Index+2].w = PACK_CAMERA_Z(CameraSpacePos.z);
	
	if( arg_7 != NULL_HANDLE && arg_8 != NULL_HANDLE )
	{
	    float3x3 TangentToWorld = mul(TexToLocal, (float3x3)LocalToWorld);
	    float4   EyeVec         = mul(In.vPosition, LocalToWorld) - CameraPos;
    		
	    Out.vTexCoords[Index+3] = float4( normalize(TangentToWorld[0]), EyeVec.x );
	    Out.vTexCoords[Index+4] = float4( normalize(TangentToWorld[1]), EyeVec.y );
	    Out.vTexCoords[Index+5] = float4( normalize(TangentToWorld[2]), EyeVec.z );
	}
#endif	
}

void DeferredLightVS(in VS_INPUT In, inout VS_OUTPUT Out, in matrix LocalToCamera, int Index)
{
	// int Index = ARG(0);

#if (bHasTexCoordOps > 0)
	Out.vTexCoords[Index] = Out.vPosition * float4(1,-1,0,1) + Out.vPosition.w;
	Out.vTexCoords[Index].xy += OneOverViewportSize.xy * Out.vTexCoords[Index].w;

	// Output camera space position
	Out.vTexCoords[Index+1].xyz = mul(float4(In.vPosition.xyz, 1), LocalToCamera).xyz;
	Out.vTexCoords[Index+1].w = Out.vPosition.w;
#endif	
}

static const int Args[6]={0,255,0,0,1,2};
void FEMU_OP_TexCoords_Func(const int ArgStart, float3 LocalNormal, const VS_INPUT In, inout VS_OUTPUT Out)
{
}

void FEMU_OP_Texture_Func(const int ArgStart, float3 LocalNormal, const VS_INPUT In, inout VS_OUTPUT Out)
{
}

VS_OUTPUT VSMain(const VS_INPUT In)
{
    VS_OUTPUT Out = (VS_OUTPUT)0;
    Out.vColor = In.vColor;
    float3 Normal = In.vNormal;
    Out.vPosition = mul(In.vPosition, LocalToScreen);
    HandleTexCoordOp_StreamIndex7(TexCoordOps[0], 0, Normal, In, Out);
    FEMU_OP_TexCoords_Func(0, Normal, In, Out);
    FEMU_OP_Texture_Func(2, Normal, In, Out);
    Out.vColor.rgb = HandleVertexLighting(Out.vColor.rgb, In.vPosition, Normal);
    return Out;
};
